#ifndef _libmyri_h_
#define _libmyri_h_

/*
 * A SW-stack independent API for accessing Myricom HW
 */

enum myri_event_type {
  MYRI_EVENT_NO_EVENT,
  MYRI_EVENT_RAW_SEND_COMPLETE,
  MYRI_EVENT_SEND_COMPLETE,
  MYRI_EVENT_RAW_RECV_COMPLETE,
  MYRI_EVENT_RECV_COMPLETE,
  MYRI_EVENT_ERROR
};

/*
 * Error codes that may be returned
 */
enum myri_error_type {
  MYRI_ERROR_FIRMWARE_UNRESPONSIVE,
  MYRI_ERROR_SOFT_SRAM_ERROR,
  MYRI_ERROR_HARD_SRAM_ERROR,
  MYRI_ERROR_UNKNOWN,
  MYRI_ERROR_CNT
};

/*
 * Myrinet line speeds
 */
enum myri_line_speed {
  MYRI_SPEED_2G,
  MYRI_SPEED_10G
};

/*
 * Information about a NIC
 */
struct myri_nic_info {
  lf_mac_addr_t mac_addr;

  char serial_no[LF_SHORT_STRING_LEN];
  char product_id[LF_SHORT_STRING_LEN];

  int num_ports;		/* number of NIC ports */
  enum myri_line_speed speed;			/* line speed */
  int num_routes;		/* number routes per destination */
  int num_routes_is_per_port;	/* boolean - if TRUE then num_routes is per */
				/* local port */
};

/*
 * NIC counters
 */
struct myri_nic_counters {
  unsigned int rx_packets;
  unsigned int tx_packets;
  unsigned int badcrcs;
};

/*
 * Event struct returned by myri_get_next_event().
 * This should be released when no longer needed via myri_release_event()
 */
struct myri_event {
  enum myri_event_type type;

  union {
    struct {
      uint32_t port;		/* port on which we received packet */
      void  *rxbuf;		/* a buffer holding the received msg */
      unsigned int rx_len;	/* length of received message */
    } raw_recv;

    struct {
      void *context;		/* context of send that just finished */
    } raw_send;

    struct {
      enum myri_error_type error;	/* error code */
    } error;
  } d;

  struct myri_event *next;
  struct myri_event *prev;
};


/*
 * Event type is returned, and context pointer is filled in with different
 * things depending on event type.
 *
 * Returns 0 on success, -1 on error with errno set.
 */

/* returns an opaque handle used by other operations, nic_id is whic
 * NIC on the host to open
 */
int myri_open(int nic_id);

/* closes this session on this NIC */
void myri_close(int nic_id);

/* return a file descriptor suitable for use in select() or poll()
 * Tell poll() or select() you want to know when READ is available, though
 * you don't actually need to read() anything.  This may be implemented
 * internally by using a pipe() and the myri thread write()ing a byte when 
 * an event occurs and read()ing it in myri_next_event().
 */
int myri_fd(int nic_id);

/* send a raw packet */
int myri_raw_send(int nic_id, int port, void *route, int route_len,
                  void *buffer, int len, void *context);

/* Get a structure describing the next event for this NIC
 * timeout is on milliseconds, and negative means infinite
 */
int myri_next_event(int nic_id, struct myri_event **mep, int timeout);

/* Called when done processing an event */
void myri_release_event(struct myri_event *mep);

int myri_mac_to_hostname(int nic_id, lf_mac_addr_t mac_addr,
			 lf_string_t hostname, int *his_nic_id);
int myri_get_nic_info(int nic_id, struct myri_nic_info *nip);
int myri_set_route_start(int nic_id);
int myri_set_route_end(int nic_id, lf_mac_addr_t mapper_mac, int map_version,
    int num_hosts);
int myri_set_route(int nic_id, lf_mac_addr_t remote_mac,
    enum lf_firmware_type fw_type,
    int remote_port, int route_num, int local_port,
    unsigned char *route, int route_len);
int myri_get_hostname(int nic_id, char *hostname);
int myri_set_hostname(int nic_id, char *hostname);
int myri_set_dflt_hostname(int nic_id, char *hostname);
int myri_get_nic_counters(int nic_id, int port,
    struct myri_nic_counters *counters);
enum lf_firmware_type  myri_firmware_type(void);
int myri_set_nic_reply_info(int nic_id, void *blob, int size);

#endif /* _libmyri_h_ */
